<?php

namespace Application\UseCases;

use Application\DTOs\ObraPublicacaoDTO;
use Domain\Entities\ObraPublicacao;
use Domain\Repositories\IObraRepository;
use Domain\Repositories\IObraPublicacaoRepository;
use DomainException;

class ManageObraPublicacao
{
    public function __construct(
        private IObraRepository $obraRepository,
        private IObraPublicacaoRepository $obraPublicacaoRepository
    ) {}

    public function execute(ObraPublicacaoDTO $dto): array
    {
        $obra = $this->obraRepository->findById($dto->obra_id);
        if (!$obra) {
            throw new DomainException('Obra não encontrada');
        }

        if ($obra->obra_intelectual !== 'PUBLICADA') {
            throw new DomainException('Apenas obras PUBLICADAS podem ter dados de publicação');
        }

        $existingPublicacao = $this->obraPublicacaoRepository->findByObraId($dto->obra_id);

        if ($existingPublicacao) {
            $existingPublicacao->update(
                $dto->editora,
                $dto->num_edicao,
                $dto->ano,
                $dto->produtora,
                $dto->local_publicacao,
                $dto->volume
            );

            $savedPublicacao = $this->obraPublicacaoRepository->update($existingPublicacao);
        } else {
            $obraPublicacao = new ObraPublicacao(
                $dto->obra_id,
                $dto->editora,
                $dto->num_edicao,
                $dto->ano,
                $dto->produtora,
                $dto->local_publicacao,
                $dto->volume
            );

            $savedPublicacao = $this->obraPublicacaoRepository->save($obraPublicacao);
        }

        return [
            'id' => $savedPublicacao->id,
            'obra_id' => $savedPublicacao->obra_id,
            'editora' => $savedPublicacao->editora,
            'num_edicao' => $savedPublicacao->num_edicao,
            'ano' => $savedPublicacao->ano,
            'produtora' => $savedPublicacao->produtora,
            'local_publicacao' => $savedPublicacao->local_publicacao,
            'volume' => $savedPublicacao->volume,
            'created_at' => $savedPublicacao->createdAt->format('Y-m-d H:i:s'),
            'updated_at' => $savedPublicacao->updatedAt?->format('Y-m-d H:i:s')
        ];
    }

    public function getByObraId(int $obra_id): ?array
    {
        $obraPublicacao = $this->obraPublicacaoRepository->findByObraId($obra_id);

        return $obraPublicacao ? $obraPublicacao->toArray() : null;
    }

    public function deleteByObraId(int $obra_id): void
    {
        $this->obraPublicacaoRepository->deleteByObraId($obra_id);
    }
}
