<?php

namespace Infrastructure\Persistence\Repositories;

use Domain\Entities\AutorObra;
use Domain\Repositories\IAutorObraRepository;
use Infrastructure\Persistence\Eloquent\Models\AutorObraModel;

class EloquentAutorObraRepository implements IAutorObraRepository
{
    public function save(AutorObra $autorObra): AutorObra
    {
        $autorObraModel = AutorObraModel::create([
            'autor_id' => $autorObra->autor_id,
            'obra_id' => $autorObra->obra_id,
            'vinculo_id' => $autorObra->vinculo_id,
            'observacoes' => $autorObra->observacoes
        ]);

        return $this->toEntity($autorObraModel);
    }

    public function findByObra(int $obra_id): array
    {
        $autorObrasModel = AutorObraModel::with(['autor', 'vinculo'])
            ->where('obra_id', $obra_id)
            ->get();

        return $autorObrasModel->map(function ($autorObraModel) {
            return $this->toEntity($autorObraModel);
        })->toArray();
    }

    public function findByAutor(int $autor_id): array
    {
        $autorObrasModel = AutorObraModel::with(['obra', 'vinculo'])
            ->where('autor_id', $autor_id)
            ->get();

        return $autorObrasModel->map(function ($autorObraModel) {
            return $this->toEntity($autorObraModel);
        })->toArray();
    }

    public function findById(int $id): ?AutorObra
    {
        $autorObraModel = AutorObraModel::with(['autor', 'obra', 'vinculo'])->find($id);

        return $autorObraModel ? $this->toEntity($autorObraModel) : null;
    }

    public function update(AutorObra $autorObra): AutorObra
    {
        $autorObraModel = AutorObraModel::findOrFail($autorObra->id);

        $autorObraModel->update([
            'vinculo_id' => $autorObra->vinculo_id,
            'observacoes' => $autorObra->observacoes
        ]);

        return $this->toEntity($autorObraModel->fresh(['autor', 'obra', 'vinculo']));
    }

    public function delete(int $id): void
    {
        AutorObraModel::destroy($id);
    }

    public function deleteByObra(int $obra_id): void
    {
        AutorObraModel::where('obra_id', $obra_id)->delete();
    }

    public function deleteByAutor(int $autor_id): void
    {
        AutorObraModel::where('autor_id', $autor_id)->delete();
    }

    private function toEntity(AutorObraModel $autorObraModel): AutorObra
    {
        return new AutorObra(
            $autorObraModel->autor_id,
            $autorObraModel->obra_id,
            $autorObraModel->vinculo_id,
            $autorObraModel->observacoes,
            $autorObraModel->id,
            null, // autor
            null, // obra
            null, // vinculo
            $autorObraModel->created_at ? \DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $autorObraModel->created_at->format('Y-m-d H:i:s')) : null,
            $autorObraModel->updated_at ? \DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $autorObraModel->updated_at->format('Y-m-d H:i:s')) : null
        );
    }
}
