<?php

namespace Infrastructure\Persistence\Repositories;

use Application\DTOs\Shared\DataResponse;
use Application\Shared\Search\PaginationMeta;
use Application\Shared\Search\SearchWithPagination;
use Domain\Entities\Autor;
use Domain\Entities\AutorRepresentanteLegal;
use Domain\Repositories\IAutorRepository;
use Infrastructure\Persistence\Eloquent\Models\AutorModel;
use Infrastructure\Persistence\Eloquent\Models\AutorRepresentanteLegalModel;
use Infrastructure\Persistence\Repositories\Shared\SearchQueryBuilder;



class EloquentAutorRepository implements IAutorRepository
{
    public function save(Autor $autor, ?AutorRepresentanteLegal $representanteLegal): Autor
    {

        if ($autor->autor_representante && !$representanteLegal) {
            $dataRepresentanteLegal = [
                'nome' => $autor->autor_representante->nome,
                'numero_identificacao' => $autor->autor_representante->numeroIdentificacao,
                'grau_parentesco' => $autor->autor_representante->grau_parentesco,
            ];

            $representanteLegal = AutorRepresentanteLegalModel::create($dataRepresentanteLegal);
            $autorModel = AutorModel::create([
                'nome' => $autor->nome,
                'email' => $autor->email,
                'provincia_id' => $autor->provincia_id,
                'numero_identificacao' => $autor->numero_identificacao,
                'telefone' => $autor->telefone,
                'data_nascimento' => $autor->data_nascimento,
                'representante_legal_id' => $representanteLegal->id,
                'pseudonimo' => $autor->pseudonimo,
                'nome_pai' => $autor->nome_pai,
                'nome_mae' => $autor->nome_mae,
                'nacionalidade_id' => $autor->nacionalidade_id,
                'bairro' => $autor->bairro,
                'endereco' => $autor->endereco,
                'municipio_id' => $autor->municipio_id,
                'ocupacao' => $autor->ocupacao,
                'naturalidade' => $autor->naturalidade,
                'celular' => $autor->celular,
                'site' => $autor->site,
            ]);
            return $this->toEntity($autorModel);
        }



        if ($representanteLegal) {
            $autorModel = AutorModel::create([
                'nome' => $autor->nome,
                'email' => $autor->email,
                'provincia_id' => $autor->provincia_id,
                'numero_identificacao' => $autor->numero_identificacao,
                'telefone' => $autor->telefone,
                'data_nascimento' => $autor->data_nascimento,
                'representante_legal_id' => $representanteLegal->id,
                'pseudonimo' => $autor->pseudonimo,
                'nome_pai' => $autor->nome_pai,
                'nome_mae' => $autor->nome_mae,
                'nacionalidade_id' => $autor->nacionalidade_id,
                'bairro' => $autor->bairro,
                'endereco' => $autor->endereco,
                'municipio_id' => $autor->municipio_id,
                'ocupacao' => $autor->ocupacao,
                'naturalidade' => $autor->naturalidade,
                'celular' => $autor->celular,
                'site' => $autor->site,
                'representante_legal_id' => $representanteLegal->id,

            ]);
            return $this->toEntity($autorModel);
        }

        $autorModel = AutorModel::create([
            'nome' => $autor->nome,
            'email' => $autor->email,
            'provincia_id' => $autor->provincia_id,
            'numero_identificacao' => $autor->numero_identificacao,
            'telefone' => $autor->telefone,
            'data_nascimento' => $autor->data_nascimento,
            'pseudonimo' => $autor->pseudonimo,
            'nome_pai' => $autor->nome_pai,
            'nome_mae' => $autor->nome_mae,
            'nacionalidade_id' => $autor->nacionalidade_id,
            'bairro' => $autor->bairro,
            'endereco' => $autor->endereco,
            'municipio_id' => $autor->municipio_id,
            'ocupacao' => $autor->ocupacao,
            'naturalidade' => $autor->naturalidade,
            'celular' => $autor->celular,
            'site' => $autor->site,
            'representante_legal_id' => null

        ]);
        return $this->toEntity($autorModel);
    }

    public function update(Autor $autor, $id): Autor
    {


        $model = AutorModel::query()->with('representanteLegal')->find($id);

        if ($autor->autor_representante) {
            $representanteLegal = AutorRepresentanteLegalModel::query()->find($model->representanteLegal->id);
            $representanteLegal->update([
                'nome' => $autor->autor_representante->nome,
                'numero_identificacao' => $autor->autor_representante->numeroIdentificacao,
                'grau_parentesco' => $autor->autor_representante->grau_parentesco,
            ]);
        }

        $model->update([
            'nome' => $autor->nome,
            'email' => $autor->email,
            'provincia_id' => $autor->provincia_id,
            'numero_identificacao' => $autor->numero_identificacao,
            'telefone' => $autor->telefone,
            'data_nascimento' => $autor->data_nascimento,
        ]);
        return $this->toEntity($model);
    }

    public function list(SearchWithPagination $input): DataResponse
    {

        $query = AutorModel::query()
            ->select(['id', 'nome', 'email', 'provincia_id', 'telefone', 'data_nascimento', 'numero_identificacao'])
            ->with(['provincia' => function ($q) {
                $q->select(['id', 'nome']); // specific fields from Provincia
            }])
            ->whereNull('deleted_at');
        $result = SearchQueryBuilder::apply($query, $input);

        if ($input->paginate->withPagination) {
            $meta = new PaginationMeta([
                'total' => $result->total(),
                'page' => $result->currentPage(),
                'perPage' => $result->perPage(),
                'sort' => $input->search->orderBy,
                'direction' => $input->search->orderByDirection,
            ]);

            return new DataResponse($result->items(), $meta);
        }
        $items = $result->get()->toArray();
        return new DataResponse($items);
    }



    public function findById(int $id): ?DataResponse
    {
        $model = AutorModel::query()
            ->with('representanteLegal')
            ->with('documentos.tipo', function ($q) {
                $q->select(['id', 'nome']); })
            ->with('nacionalidade', function ($q) {
                $q->select(['id', 'nome']); 
            })
            ->with('provincia', function ($q) {
                $q->select(['id', 'nome']);
            })
            ->with('municipio', function ($q) {
                $q->select(['id', 'nome']); }
                )
            ->whereNull('deleted_at')
            ->find($id);

        if (!$model) {
            return null;
        }

        return new DataResponse($model);
    }


    public function delete(int $id): void
    {
        $model = AutorModel::query()->find($id);
        $model->delete();
    }


    private function toEntity(AutorModel $data): Autor
    {



        return new Autor(
            id: $data->id,
            nome: $data->nome,
            email: $data->email,
            provincia_id: $data->provincia_id,
            autor_representante: $data->representante_legal,
            data_nascimento: $data->data_nascimento,
            telefone: $data->telefone,
            nacionalidade_id: $data->nacionalidade_id,
            numero_identificacao: $data->numero_identificacao,
            pseudonimo: $data->pseudonimo,
            naturalidade: $data->naturalidade,
            nome_pai: $data->nome_pai,
            nome_mae: $data->nome_mae,
            ocupacao: $data->ocupacao,
            endereco: $data->endereco,
            bairro: $data->bairro,
            municipio_id: $data->municipio_id,
            celular: $data->celular,
            site: $data->site,

        );
    }

    public function findRepresentanteLegal(?string $numero_identificacao): ?AutorRepresentanteLegal
    {
        $model = AutorRepresentanteLegalModel::query()
            ->where('numero_identificacao', $numero_identificacao)
            ->first();

        if (!$model) {
            return null;
        }

        return new AutorRepresentanteLegal(
            id: $model->id,
            nome: $model->nome,
            numeroIdentificacao: $model->numero_identificacao,
            grauParentesco: $model->grau_parentesco,
        );
    }
}
