<?php

namespace Infrastructure\Persistence\Repositories;

use Domain\Entities\Obra;
use Domain\Repositories\IObraRepository;
use Infrastructure\Persistence\Eloquent\Models\ObraModel;

class EloquentObraRepository implements IObraRepository
{
    public function save(Obra $obra): Obra
    {
        $obraModel = ObraModel::create([
            'titulo' => $obra->titulo,
            'numero_certificado' => $obra->numero_certificado,
            'data_registo' => $obra->data_registo->format('Y-m-d'),
            'provincia_id' => $obra->provincia_id,
            'genero_id' => $obra->genero_id,
            'suporte' => $obra->suporte,
            'classificacao_id' => $obra->classificacao_id,
            'obra_intelectual' => $obra->obra_intelectual,
            'descricao' => $obra->descricao,
            'observacoes' => $obra->observacoes
        ]);

        return $this->toEntity($obraModel);
    }

    public function findById(int $id): ?Obra
    {
        $obraModel = ObraModel::with(['genero', 'classificacao'])->find($id);

        return $obraModel ? $this->toEntity($obraModel) : null;
    }

    public function findByCertificateNumber(string $numero_certificado): ?Obra
    {
        $obraModel = ObraModel::with(['genero', 'classificacao'])
            ->where('numero_certificado', $numero_certificado)
            ->first();

        return $obraModel ? $this->toEntity($obraModel) : null;
    }

    public function findAll(): array
    {
        $obrasModel = ObraModel::with(['genero', 'classificacao'])->get();

        return $obrasModel->map(function ($obraModel) {
            return $this->toEntity($obraModel);
        })->toArray();
    }

    public function update(Obra $obra): Obra
    {
        $obraModel = ObraModel::findOrFail($obra->id);

        $obraModel->update([
            'titulo' => $obra->titulo,
            'numero_certificado' => $obra->numero_certificado,
            'data_registo' => $obra->data_registo->format('Y-m-d'),
            'provincia_id' => $obra->provincia_id,
            'genero_id' => $obra->genero_id,
            'suporte' => $obra->suporte,
            'classificacao_id' => $obra->classificacao_id,
            'obra_intelectual' => $obra->obra_intelectual,
            'descricao' => $obra->descricao,
            'observacoes' => $obra->observacoes
        ]);

        return $this->toEntity($obraModel->fresh(['genero', 'classificacao']));
    }

    public function delete(int $id): void
    {
        ObraModel::destroy($id);
    }

    public function findByAuthor(int $autor_id): array
    {
        $obrasModel = ObraModel::with(['genero', 'classificacao'])
            ->whereHas('autores', function ($query) use ($autor_id) {
                $query->where('autor_id', $autor_id);
            })
            ->get();

        return $obrasModel->map(function ($obraModel) {
            return $this->toEntity($obraModel);
        })->toArray();
    }

    private function toEntity(ObraModel $obraModel): Obra
    {
        return new Obra(
            $obraModel->id,
            $obraModel->titulo,
            $obraModel->numero_certificado,
            $obraModel->data_registo,
            $obraModel->provincia_id,
            $obraModel->genero_id,
            $obraModel->suporte,
            $obraModel->classificacao_id,
            $obraModel->obra_intelectual,
            $obraModel->descricao,
            $obraModel->observacoes,
            $obraModel->created_at ? \DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $obraModel->created_at->format('Y-m-d H:i:s')) : null,
            $obraModel->updated_at ? \DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $obraModel->updated_at->format('Y-m-d H:i:s')) : null
        );
    }
}
