<?php

namespace Infrastructure\Persistence\Repositories;

use Application\DTOs\Shared;
use Application\DTOs\Shared\DataResponse;
use Application\Shared\Search\PaginationMeta;
use Application\Shared\Search\SearchWithPagination;
use Domain\Entities\Request;
use Domain\Repositories\RequestRepositoryInterface;
use Infrastructure\Persistence\Eloquent\Models\RequestModel;
use Infrastructure\Persistence\Repositories\Shared\SearchQueryBuilder;

class EloquentRequestRepository implements RequestRepositoryInterface
{
    public function save(Request $request): Request
    {
        $data = [
            'certificate_number' => $request->certificate_number,
            'registration_date' => $request->registration_date,
            'province' => $request->province,
            'author_name' => $request->author_name,
            'work_title' => $request->work_title,
            'genre' => $request->genre,
            'support' => $request->support,
            'classification' => $request->classification,
            'is_published' => $request->is_published,
        ];
        $model = RequestModel::create($data);

        return $this->toEntity($model);
    }

    public function update(Request $request): Request
    {
        $data = [
            'certificate_number' => $request->certificate_number,
            'registration_date' => $request->registration_date,
            'province' => $request->province,
            'author_name' => $request->author_name,
            'work_title' => $request->work_title,
            'genre' => $request->genre,
            'support' => $request->support,
            'classification' => $request->classification,
            'is_published' => $request->is_published,
        ];

    //         dd([
    //     'ID from Request Entity' => $request->id,
    //     'Type of ID' => gettype($request->id),
    //     'Target Table' => (new RequestModel())->getTable(),
    // ]); 
        $model = RequestModel::findOrFail($request->id);

        $model->update($data);

        return $this->toEntity($model);
    }

    public function findById(int $id): ?Request
    {
        $m = RequestModel::find($id);

        return $m ? $this->toEntity($m) : null;
    }

    public function findByEmail(string $email): ?Request
    {
        $m = RequestModel::where('email', $email)->first();

        return $m ? $this->toEntity($m) : null;
    }

    public function list(SearchWithPagination $input): DataResponse
    {
        $query = RequestModel::query();
        $result = SearchQueryBuilder::apply($query, $input);

        if($input ->paginate->withPagination){
               $meta = new PaginationMeta([
                'total' => $result->total(),
                'page' => $result->currentPage(),
                'perPage' => $result->perPage(),
                'sort' => $input->search->orderBy,
                'direction' => $input->search->orderByDirection,
            ]);

            return new DataResponse($result->items(), $meta);
        }
        $items = $result->get()->toArray();
        return new DataResponse($items);
    }

    public function delete(int $id): void
    {
        $request = RequestModel::find($id);

        $request->delete();
    }

    private function toEntity(RequestModel $data): Request
    {
        return new Request(
            id: $data->id,
            certificate_number: $data->certificate_number ?? '',  // ✅ Garantir string
            registration_date: $data->registration_date instanceof \DateTime
                ? $data->registration_date
                : new \DateTime($data->registration_date),  // ✅ Garantir DateTime
            province: $data->province ?? '',
            author_name: $data->author_name ?? '',
            work_title: $data->work_title ?? '',
            genre: $data->genre ?? '',
            support: $data->support ?? '',
            classification: $data->classification ?? '',
            is_published: (bool) $data->is_published  // ✅ Garantir boolean
        );
    }
}
