<?php

namespace Infrastructure\Persistence\Repositories\Shared;

use Application\Shared\Search\SearchWithPagination;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Application\Shared\Search\Operator;

class SearchQueryBuilder
{
    /**
     * @param Builder $query
     * @param SearchWithPagination $input
     * @return Builder|LengthAwarePaginator
     */
    public static function apply(Builder $query, SearchWithPagination $input)
    {
        $search = $input->search;
        $paginate = $input->paginate;

        // SearchValue em múltiplas colunas
        if (!empty($search->searchValue) && !empty($search->searchBy)) {
            $query->where(function ($q) use ($search) {
                foreach ($search->searchBy as $index => $column) {
                    $method = $index === 0 ? 'where' : 'orWhere';
                    $q->{$method}($column, 'LIKE', '%' . $search->searchValue . '%');
                }
            });
        }

        // Filtros dinâmicos
        if (!empty($search->filters)) {
            foreach ($search->filters as $field => $filter) {
                $operation = $filter['operation']; // Operator enum
                $value = $filter['value'];

                if (!$operation instanceof Operator) {
                    // tentar converter (tolerância)
                    try {
                        $operation = Operator::from($operation);
                    } catch (\ValueError $e) {
                        continue; // pula operador inválido
                    }
                }

                switch ($operation) {
                    case Operator::LIKE:
                        $query->where($field, 'LIKE', '%' . $value . '%');
                        break;

                    case Operator::EQ:
                        $query->where($field, '=', $value);
                        break;

                    case Operator::NEQ:
                        $query->where($field, '!=', $value);
                        break;

                    case Operator::GTE:
                        $query->where($field, '>=', $value);
                        break;

                    case Operator::LTE:
                        $query->where($field, '<=', $value);
                        break;

                    case Operator::IN:
                        if (is_string($value)) {
                            $arr = array_map('trim', explode(',', $value));
                        } elseif (is_array($value)) {
                            $arr = $value;
                        } else {
                            $arr = [$value];
                        }
                        if (!empty($arr)) {
                            $query->whereIn($field, $arr);
                        }
                        break;

                    case Operator::BETWEEN:
                        if (is_string($value) && str_contains($value, ',')) {
                            $parts = array_map('trim', explode(',', $value));
                            if (count($parts) === 2) {
                                $query->whereBetween($field, $parts);
                            }
                        } elseif (is_array($value) && count($value) === 2) {
                            $query->whereBetween($field, $value);
                        }
                        break;

                    default:
                        // operador não tratado - ignora
                        break;
                }
            }
        }

        // Ordenação
        if (!empty($search->orderBy)) {
            $query->orderBy($search->orderBy, $search->orderByDirection ?? 'asc');
        }

        // Paginação
        if ($paginate->withPagination) {
            return $query->paginate($paginate->perPage, ['*'], 'page', $paginate->page);
        }

        return $query;
    }
}
