<?php

namespace UI\Http\Controllers;

use App\Http\Controllers\Controller;
use Domain\Entities\Genero;
use Domain\Repositories\IGeneroRepositoryInterface;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use UI\Http\Requests\StoreGeneroRequest;

/**
 * @OA\Tag(
 *     name="Géneros",
 *     description="Endpoints para gestão de géneros"
 * )
 */
class GeneroController extends Controller
{
    public function __construct(
        private IGeneroRepositoryInterface $repo
    ) {
    }

    /**
     * @OA\Get(
     *     path="/api/v1/generos",
     *     tags={"Géneros"},
     *     summary="Listar géneros com paginação e filtro opcional",
     *     description="Retorna uma lista de géneros. É possível definir o número da página e a quantidade de itens por página.",
     *     security={{"bearerAuth":{}}},
     *
     *     @OA\Parameter(
     *         name="search",
     *         in="query",
     *         description="Texto opcional para filtrar pela designação do género",
     *         required=false,
     *
     *         @OA\Schema(type="string", example="literatura")
     *     ),
     *
     *      @OA\Parameter(
     *         name="paginate",
     *         in="query",
     *         description="Define se deve paginar (true/false)",
     *         required=false,
     *
     *         @OA\Schema(type="boolean", example=true)
     *     ),
     *
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         description="Número da página atual (para paginação)",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\Parameter(
     *         name="perPage",
     *         in="query",
     *         description="Quantidade de resultados por página",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=10)
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Lista de géneros retornada com sucesso",
     *
     *         @OA\JsonContent(
     *             type="object",
     *
     *             @OA\Property(property="data", type="array",
     *
     *                 @OA\Items(
     *
     *                     @OA\Property(property="id", type="string", example="a5f7c8e9-23ab-4e7c-9b10-dc9876543210"),
     *                     @OA\Property(property="designacao", type="string", example="Literatura"),
     *                     @OA\Property(property="created_at", type="string", format="date-time", example="2025-10-20T23:40:46Z"),
     *                     @OA\Property(property="updated_at", type="string", format="date-time", example="2025-10-20T23:40:46Z")
     *                 )
     *             ),
     *             @OA\Property(property="meta", type="object",
     *                 @OA\Property(property="current_page", type="integer", example=1),
     *                 @OA\Property(property="per_page", type="integer", example=10),
     *                 @OA\Property(property="total", type="integer", example=25),
     *                 @OA\Property(property="last_page", type="integer", example=3)
     *             )
     *         )
     *     )
     * )
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $page = (int) $request->input('page', 1);
            $perPage = (int) $request->input('perPage', 20);
            $search = $request->input('search') ?? $request->input('nome') ?? $request->input('descricao');
            $paginate = true;

            $useCase = app(\Application\UseCases\ListGeneros::class);
            $result = $useCase->execute($search, $perPage, $page, $paginate);

            return response()->json([
                'success' => true,
                'data' => $result,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => count($result),
                    'last_page' => ceil(count($result) / $perPage)
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/generos",
     *     tags={"Géneros"},
     *     summary="Criar um novo género",
     *     description="Cria um novo género com base na designação fornecida.",
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\JsonContent(
     *             required={"designacao"},
     *
     *             @OA\Property(property="designacao", type="string", example="Literatura")
     *         )
     *     ),
     *
     *     @OA\Response(response=201, description="Género criado com sucesso"),
     *     @OA\Response(response=400, description="Requisição inválida")
     * )
     */
    public function store(StoreGeneroRequest $request): JsonResponse
    {
        $genero = new Genero(designacao: $request->input('designacao'));
        $this->repo->save($genero);
        return response()->json($genero, 201);
    }

    /**
     * @OA\Get(
     *     path="/api/v1/generos/{id}",
     *     tags={"Géneros"},
     *     summary="Obter um género específico",
     *
     *     @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer", example=1)),
     *
     *     @OA\Response(response=200, description="Género retornado com sucesso"),
     *     @OA\Response(response=404, description="Género não encontrado")
     * )
     */
    public function show(int $id): JsonResponse
    {
        $genero = $this->repo->findById($id);
        if (!$genero) {
            return response()->json(['message' => 'Género não encontrado'], 404);
        }
        return response()->json($genero, 200);
    }

    /**
     * @OA\Put(
     *     path="/api/v1/generos/{id}",
     *     tags={"Géneros"},
     *     summary="Atualizar um género existente",
     *
     *     @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer", example=1)),
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\JsonContent(
     *             required={"designacao"},
     *
     *             @OA\Property(property="designacao", type="string", example="Poesia")
     *         )
     *     ),
     *
     *     @OA\Response(response=200, description="Género atualizado com sucesso"),
     *     @OA\Response(response=404, description="Género não encontrado")
     * )
     */
    public function update(int $id, Request $request): JsonResponse
    {
        $genero = $this->repo->findById($id);

        if (!$genero) {
            return response()->json(['message' => 'Género não encontrado'], 404);
        }

        $genero->designacao = $request->input('designacao');
        $this->repo->save($genero);

        return response()->json($genero, 200);
    }

    /**
     * @OA\Delete(
     *     path="/api/v1/generos/{id}",
     *     tags={"Géneros"},
     *     summary="Eliminar um género",
     *     description="Remove (soft delete) um género existente pelo ID.",
     *
     *     @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer", example=1)),
     *
     *     @OA\Response(response=204, description="Género eliminado com sucesso"),
     *     @OA\Response(response=404, description="Género não encontrado")
     * )
     */
    public function destroy(int $id): JsonResponse
    {
        $genero = $this->repo->findById($id);

        if (!$genero) {
            return response()->json(['message' => 'Género não encontrado'], 404);
        }

        $this->repo->delete($genero->id);

        return response()->json(null, 204);
    }
}
