<?php

namespace UI\Http\Controllers;

use App\Http\Controllers\Controller;
use Application\DTOs\UserData;
use Application\UseCases\CreateUser;
use Application\UseCases\UpdateUser;
use Domain\Repositories\UserRepositoryInterface;
use UI\Http\Requests\StoreUserRequest;
use UI\Http\Requests\UpdateUserRequest;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * @OA\Tag(name="Users", description="Operações CRUD de utilizadores")
 */
class UserController extends Controller
{
    public function __construct(
        private UserRepositoryInterface $repo,
        private CreateUser $createUser,
        private UpdateUser $updateUser
    ) {}

    /**
     * @OA\Get(
     *   path="/api/v1/users",
     *   tags={"Users"},
     *   security={{"bearerAuth":{}}},
     *   summary="Listar usuários com paginação e filtros",
     *   description="Retorna uma lista paginada de usuários com filtros e ordenação",
     *   @OA\Parameter(
     *     name="page",
     *     in="query",
     *     description="Número da página para paginação",
     *     required=false,
     *     @OA\Schema(type="integer", default=1)
     *   ),
     *   @OA\Parameter(
     *     name="perPage",
     *     in="query",
     *     description="Número de itens por página",
     *     required=false,
     *     @OA\Schema(type="integer", default=20)
     *   ),
     *   @OA\Parameter(
     *     name="search",
     *     in="query",
     *     description="Filtrar por nome ou email do usuário (busca parcial)",
     *     required=false,
     *     @OA\Schema(type="string")
     *   ),
     *   @OA\Parameter(
     *     name="name",
     *     in="query",
     *     description="Filtrar por nome do usuário (busca parcial)",
     *     required=false,
     *     @OA\Schema(type="string")
     *   ),
     *   @OA\Parameter(
     *     name="email",
     *     in="query",
     *     description="Filtrar por email do usuário (busca parcial)",
     *     required=false,
     *     @OA\Schema(type="string")
     *   ),
     *   @OA\Response(
     *     response=200,
     *     description="Lista de usuários retornada com sucesso",
     *     @OA\JsonContent(
     *       @OA\Property(property="success", type="boolean", example=true),
     *       @OA\Property(
     *         property="data",
     *         type="array",
     *         @OA\Items(
     *           @OA\Property(property="id", type="integer", example=1),
     *           @OA\Property(property="name", type="string", example="João Silva"),
     *           @OA\Property(property="email", type="string", example="joao@example.com"),
     *           @OA\Property(property="role", type="string", example="admin"),
     *           @OA\Property(property="created_at", type="string", format="date-time"),
     *           @OA\Property(property="updated_at", type="string", format="date-time")
     *         )
     *       ),
     *       @OA\Property(
     *         property="pagination",
     *         type="object",
     *         @OA\Property(property="current_page", type="integer", example=1),
     *         @OA\Property(property="per_page", type="integer", example=20),
     *         @OA\Property(property="total", type="integer", example=10),
     *         @OA\Property(property="last_page", type="integer", example=1)
     *       )
     *     )
     *   ),
     *   @OA\Response(
     *     response=500,
     *     description="Erro interno do servidor",
     *     @OA\JsonContent(
     *       @OA\Property(property="success", type="boolean", example=false),
     *       @OA\Property(property="message", type="string", example="Erro ao listar usuários")
     *     )
     *   )
     * )
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $page = (int) $request->input('page', 1);
            $perPage = (int) $request->input('perPage', 20);
            $search = $request->input('search') ?? $request->input('name') ?? $request->input('email');

            $offset = ($page - 1) * $perPage;

            $users = $this->repo->list($perPage, $offset);

            if ($search) {
                $users = array_filter($users, function($user) use ($search) {
                    return stripos($user->name, $search) !== false ||
                           stripos($user->email, $search) !== false;
                });
            }

            $payload = array_map(fn($u) => [
                'id' => $u->id,
                'name' => $u->name,
                'email' => $u->email,
                'role' => $u->role,
                'created_at' => $u->created_at,
                'updated_at' => $u->updated_at
            ], $users);

            return response()->json([
                'success' => true,
                'data' => $payload,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => count($payload),
                    'last_page' => ceil(count($payload) / $perPage)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/api/v1/users",
     *   tags={"Users"},
     *   security={{"bearerAuth":{}}},
     *   @OA\RequestBody(
     *     required=true,
     *     @OA\JsonContent(
     *       required={"name","email","password","password_confirmation"},
     *       @OA\Property(property="name", type="string"),
     *       @OA\Property(property="email", type="string"),
     *       @OA\Property(property="password", type="string"),
     *       @OA\Property(property="password_confirmation", type="string"),
     *       @OA\Property(property="role", type="string")
     *     )
     *   ),
     *   @OA\Response(response=201, description="User created")
     * )
     */
    public function store(StoreUserRequest $request): JsonResponse
    {
        $dto = new UserData($request->name, $request->email, $request->password, $request->role ?? 'user');
        $user = $this->createUser->execute($dto);

        return response()->json(['id' => $user->id, 'name' => $user->name, 'email' => $user->email], 201);
    }

    /**
     * @OA\Get(
     *   path="/api/v1/users/{id}",
     *   tags={"Users"},
     *   security={{"bearerAuth":{}}},
     *   @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer")),
     *   @OA\Response(response=200, description="Get user"),
     *   @OA\Response(response=404, description="Not found")
     * )
     */
    public function show(int $id): JsonResponse
    {
        $u = $this->repo->findById($id);
        if (!$u) return response()->json(['message' => 'Not found'], 404);

        return response()->json(['id' => $u->id, 'name' => $u->name, 'email' => $u->email, 'role' => $u->role]);
    }

    /**
     * @OA\Put(
     *   path="/api/v1/users/{id}",
     *   tags={"Users"},
     *   security={{"bearerAuth":{}}},
     *   @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer")),
     *   @OA\RequestBody(
     *     required=true,
     *     @OA\JsonContent(
     *         type="object",
     *         required={"name", "email"},
     *         @OA\Property(property="name", type="string", example="João Silva"),
     *         @OA\Property(property="email", type="string", format="email", example="joao@email.com"),
     *         @OA\Property(property="password", type="string", format="password", nullable=true, example="secret123"),
     *         @OA\Property(property="password_confirmation", type="string"),
     *         @OA\Property(property="role", type="string", example="user", description="Perfil do utilizador (padrão: user)")
     *   )),
     *   @OA\Response(response=200, description="Updated"),
     * )
     */
    public function update(UpdateUserRequest $request, int $id): JsonResponse
    {
        $dto = new UserData($request->name, $request->email, $request->password ?: null, $request->role ?? 'user');
        $user = $this->updateUser->execute($id, $dto);

        if (!$user) return response()->json(['message' => 'Not found'], 404);

        return response()->json(['id' => $user->id, 'name' => $user->name, 'email' => $user->email, 'role' => $user->role]);
    }

    /**
     * @OA\Delete(
     *   path="/api/v1/users/{id}",
     *   tags={"Users"},
     *   security={{"bearerAuth":{}}},
     *   @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer")),
     *   @OA\Response(response=204, description="Deleted")
     * )
     */
    public function destroy(int $id): JsonResponse
    {
        $u = $this->repo->findById($id);
        if (!$u) return response()->json(['message' => 'Not found'], 404);

        $this->repo->delete($id);
        return response()->json(null, 204);
    }
}
