<?php

namespace UI\Http\Controllers;

use Application\UseCases\{
    CreateVinculo,
    DeleteVinculo,
    ListVinculos,
    UpdateVinculo
};
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;

/**
 * @OA\Tag(
 *     name="Vínculos",
 *     description="Endpoints para gestão de vínculos"
 * )
 */
class VinculoController extends Controller
{
    public function __construct(
        private CreateVinculo $createVinculo,
        private ListVinculos $listVinculos,
        private UpdateVinculo $updateVinculo,
        private DeleteVinculo $deleteVinculo
    ) {
    }

    /**
     * @OA\Get(
     *     path="/api/v1/vinculos",
     *     tags={"Vínculos"},
     *     summary="Listar vínculos com paginação e filtro opcional",
     *     description="Retorna uma lista de vínulos. É possível filtrar pela designação e controlar paginação.",
     *
     *     @OA\Parameter(
     *         name="search",
     *         in="query",
     *         description="Texto opcional para filtrar pela designação do vínculo",
     *         required=false,
     *
     *         @OA\Schema(type="string", example="efetivo")
     *     ),
     *
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         description="Número da página actual (para paginação)",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\Parameter(
     *         name="perPage",
     *         in="query",
     *         description="Quantidade de resultados por página",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=10)
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Lista de vínculos retornada com sucesso",
     *
     *         @OA\JsonContent(
     *             type="object",
     *
     *             @OA\Property(property="data", type="array",
     *
     *                 @OA\Items(
     *
     *                     @OA\Property(property="id", type="integer", example=1),
     *                     @OA\Property(property="designacao", type="string", example="Efetivo"),
     *                     @OA\Property(property="created_at", type="string", format="date-time", example="2025-10-21T10:00:00Z"),
     *                     @OA\Property(property="updated_at", type="string", format="date-time", example="2025-10-21T10:00:00Z")
     *                 )
     *             ),
     *             @OA\Property(property="meta", type="object",
     *                 @OA\Property(property="current_page", type="integer", example=1),
     *                 @OA\Property(property="per_page", type="integer", example=10),
     *                 @OA\Property(property="total", type="integer", example=25),
     *                 @OA\Property(property="last_page", type="integer", example=3)
     *             )
     *         )
     *     )
     * )
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $page = (int) $request->input('page', 1);
            $perPage = (int) $request->input('perPage', 20);
            $search = $request->input('search') ?? $request->input('nome') ?? $request->input('descricao');
            $paginate = true;

            $vinculos = $this->listVinculos->execute($perPage, $page, $search, $paginate);

            return response()->json([
                'success' => true,
                'data' => $vinculos,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => count($vinculos),
                    'last_page' => ceil(count($vinculos) / $perPage)
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/vinculos",
     *     tags={"Vínculos"},
     *     summary="Criar um novo vínculo",
     *     description="Cria um vínculo com base na designação fornecida.",
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\JsonContent(
     *             required={"designacao"},
     *
     *             @OA\Property(property="designacao", type="string", example="Efetivo")
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=201,
     *         description="Vínculo criado com sucesso",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="id", type="integer", example=1),
     *             @OA\Property(property="designacao", type="string", example="Efetivo"),
     *             @OA\Property(property="created_at", type="string", format="date-time"),
     *             @OA\Property(property="updated_at", type="string", format="date-time")
     *         )
     *     )
     * )
     */
    public function store(Request $request): JsonResponse
    {
        $data = $request->validate(['designacao' => 'required|string|max:255']);
        $vinculo = $this->createVinculo->execute($data['designacao']);
        return response()->json($vinculo, 201);
    }

    /**
     * @OA\Put(
     *     path="/api/v1/vinculos/{id}",
     *     tags={"Vínculos"},
     *     summary="Actualizar vínculo existente",
     *     description="Actualiza a designação de um vínculo pelo seu ID.",
     *
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID do vínculo",
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\JsonContent(
     *             required={"designacao"},
     *
     *             @OA\Property(property="designacao", type="string", example="Contratado")
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Vínculo actualizado com sucesso",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="id", type="integer", example=1),
     *             @OA\Property(property="designacao", type="string", example="Contratado")
     *         )
     *     ),
     *
     *     @OA\Response(response=404, description="Vínculo não encontrado")
     * )
     */
    public function update(Request $request, int $id): JsonResponse
    {
        $data = $request->validate(['designacao' => 'required|string|max:255']);
        $vinculo = $this->updateVinculo->execute($id, $data['designacao']);
        if (!$vinculo) {
            return response()->json(['message' => 'Vínculo não encontrado'], 404);
        }
        return response()->json($vinculo, 200);
    }

    /**
     * @OA\Delete(
     *     path="/api/v1/vinculos/{id}",
     *     tags={"Vínculos"},
     *     summary="Eliminar vínculo",
     *     description="Elimina um vínculo pelo seu ID.",
     *
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID do vínculo",
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\Response(response=200, description="Vínculo eliminado com sucesso")
     * )
     */
    public function destroy(int $id): JsonResponse
    {
        $deleted = $this->deleteVinculo->execute($id);
        return response()->json(['deleted' => $deleted]);
    }
}
